<?php
namespace App\Controllers;

use App\Core\Controller;
use App\Core\Helpers;
use App\Core\Marketplace;
use App\Core\Validator;

class PanelController extends Controller
{
    private Marketplace $market;

    public function __construct(array $config, $storage, $auth, $security)
    {
        parent::__construct($config, $storage, $auth, $security);
        $this->market = new Marketplace($storage);
    }

    public function dashboard(): void
    {
        $this->auth->requireRole(['user', 'admin', 'superadmin']);
        $user = $this->auth->user();
        $items = array_filter($this->market->allListings(), fn($x) => ($x['ownerId'] ?? '') === $user['id']);
        $notifications = $this->storage->readJson('users/' . $user['id'] . '/notifications.json', ['items' => []])['items'] ?? [];
        $favorites = $this->storage->readJson('users/' . $user['id'] . '/favorites.json', ['items' => []])['items'] ?? [];
        $views = array_sum(array_map(fn($x) => (int)($x['views'] ?? 0), $items));
        $this->view('panel/dashboard', [
            'title' => 'Panel',
            'user' => $user,
            'items' => $items,
            'notifications' => $notifications,
            'favoritesCount' => count($favorites),
            'viewsTotal' => $views,
        ]);
    }

    public function profileForm(): void
    {
        $this->auth->requireRole(['user', 'admin', 'superadmin']);
        $this->view('panel/profile', ['title' => 'Perfil', 'user' => $this->auth->user()]);
    }

    public function profileSave(): void
    {
        $this->auth->requireRole(['user', 'admin', 'superadmin']);
        if (!$this->security->verifyCsrf($_POST['_csrf'] ?? null)) Helpers::redirect('/panel/perfil');
        $user = $this->auth->user();
        $user['name'] = trim($_POST['name'] ?? $user['name']);
        $user['phone'] = trim($_POST['phone'] ?? '');
        $user['whatsapp'] = trim($_POST['whatsapp'] ?? '');
        $user['location'] = trim($_POST['location'] ?? '');
        $this->storage->writeJson('users/' . $user['id'] . '/profile.json', $user);

        if (!empty($_FILES['avatar']['name'])) {
            $avatar = $this->handleImage($_FILES['avatar'], 'users/' . $user['id'] . '/avatar', 1)[0] ?? '';
            if ($avatar) {
                $user['avatar'] = $avatar;
                $this->storage->writeJson('users/' . $user['id'] . '/profile.json', $user);
            }
        }
        Helpers::flash('success', 'Perfil actualizado.');
        Helpers::redirect('/panel/perfil');
    }

    public function changePasswordForm(): void
    {
        $this->auth->requireRole(['user', 'admin', 'superadmin']);
        $this->view('panel/change_password', ['title' => 'Cambiar contraseña']);
    }

    public function changePasswordSave(): void
    {
        $this->auth->requireRole(['user', 'admin', 'superadmin']);
        if (!$this->security->verifyCsrf($_POST['_csrf'] ?? null)) Helpers::redirect('/panel/cambiar-clave');
        $user = $this->auth->user();
        $sec = $this->storage->readJson('users/' . $user['id'] . '/security.json', []);
        if (!password_verify($_POST['current_password'] ?? '', $sec['passwordHash'] ?? '')) {
            Helpers::flash('error', 'Contraseña actual incorrecta.');
            Helpers::redirect('/panel/cambiar-clave');
        }
        if (strlen($_POST['new_password'] ?? '') < 8) {
            Helpers::flash('error', 'La nueva contraseña es muy corta.');
            Helpers::redirect('/panel/cambiar-clave');
        }
        $sec['passwordHash'] = password_hash($_POST['new_password'], PASSWORD_BCRYPT);
        $sec['lastPasswordChange'] = time();
        $this->storage->writeJson('users/' . $user['id'] . '/security.json', $sec);
        Helpers::flash('success', 'Contraseña actualizada.');
        Helpers::redirect('/panel');
    }

    public function listings(): void
    {
        $this->auth->requireRole(['user', 'admin', 'superadmin']);
        $user = $this->auth->user();
        $items = array_values(array_filter($this->market->allListings(), fn($x) => ($x['ownerId'] ?? '') === $user['id']));
        $this->view('panel/listings', ['title' => 'Mis publicaciones', 'items' => $items]);
    }

    public function listingForm(): void
    {
        $this->auth->requireRole(['user', 'admin', 'superadmin']);
        $this->view('panel/listing_form', [
            'title' => 'Nueva publicación',
            'categories' => $this->market->getCategories(),
            'item' => null,
        ]);
    }

    public function listingCreate(): void
    {
        $this->auth->requireRole(['user', 'admin', 'superadmin']);
        if (!$this->security->verifyCsrf($_POST['_csrf'] ?? null)) Helpers::redirect('/panel/publicacion/nueva');

        $v = (new Validator())
            ->required('title', $_POST['title'] ?? '', 'Título requerido')
            ->required('category', $_POST['category'] ?? '', 'Categoría requerida')
            ->required('condition', $_POST['condition'] ?? '', 'Condición requerida')
            ->numeric('price', $_POST['price'] ?? '', 'Precio inválido');
        if ($v->fails()) {
            Helpers::flash('error', implode(' | ', $v->errors()));
            Helpers::redirect('/panel/publicacion/nueva');
        }

        $user = $this->auth->user();
        $id = Helpers::randomId('l');
        $images = $this->handleImage($_FILES['images'] ?? null, 'listings/' . $id . '/images', $this->config['app']['max_images_per_listing']);

        $item = [
            'id' => $id,
            'ownerId' => $user['id'],
            'title' => trim($_POST['title']),
            'slug' => Helpers::slug($_POST['title']),
            'category' => trim($_POST['category']),
            'subcategory' => trim($_POST['subcategory'] ?? ''),
            'condition' => trim($_POST['condition']),
            'price' => (float)($_POST['price'] ?? 0),
            'currency' => $this->config['app']['currency'],
            'description' => trim($_POST['description'] ?? ''),
            'tags' => array_values(array_filter(array_map('trim', explode(',', $_POST['tags'] ?? '')))),
            'location' => trim($_POST['location'] ?? ''),
            'images' => $images,
            'status' => 'pending',
            'featured' => false,
            'priority' => 0,
            'views' => 0,
            'offerType' => $_POST['offerType'] ?? 'venta',
            'shipping' => isset($_POST['shipping']) ? 1 : 0,
            'sellerVerified' => (bool)($user['verified'] ?? false),
            'createdAt' => time(),
            'updatedAt' => time(),
            'moderation' => ['approvedBy' => null, 'rejectedReason' => null],
        ];

        $this->market->saveListing($item);
        $this->storage->appendLog('app.log', 'LISTING_CREATED ' . $item['id'] . ' by ' . $user['id']);
        Helpers::flash('success', 'Publicación creada y enviada a moderación.');
        Helpers::redirect('/panel/publicaciones');
    }

    public function listingEditForm(): void
    {
        $this->auth->requireRole(['user', 'admin', 'superadmin']);
        $id = $_GET['id'] ?? '';
        $item = $this->market->getListing($id);
        $user = $this->auth->user();
        if (!$item || ($item['ownerId'] !== $user['id'] && !$this->auth->hasRole(['admin', 'superadmin']))) {
            http_response_code(403);
            exit('Sin acceso');
        }
        $this->view('panel/listing_form', ['title' => 'Editar publicación', 'categories' => $this->market->getCategories(), 'item' => $item]);
    }

    public function listingEditSave(): void
    {
        $this->auth->requireRole(['user', 'admin', 'superadmin']);
        if (!$this->security->verifyCsrf($_POST['_csrf'] ?? null)) Helpers::redirect('/panel/publicaciones');
        $id = $_GET['id'] ?? '';
        $item = $this->market->getListing($id);
        $user = $this->auth->user();
        if (!$item || ($item['ownerId'] !== $user['id'] && !$this->auth->hasRole(['admin', 'superadmin']))) {
            http_response_code(403);
            exit('Sin acceso');
        }

        $allowFull = (($item['status'] ?? '') === 'pending') || $this->auth->hasRole(['admin', 'superadmin']);
        if ($allowFull) {
            $item['title'] = trim($_POST['title'] ?? $item['title']);
            $item['slug'] = Helpers::slug($item['title']);
            $item['category'] = trim($_POST['category'] ?? $item['category']);
            $item['subcategory'] = trim($_POST['subcategory'] ?? $item['subcategory']);
            $item['condition'] = trim($_POST['condition'] ?? $item['condition']);
            $item['price'] = (float)($_POST['price'] ?? $item['price']);
            $item['description'] = trim($_POST['description'] ?? $item['description']);
            $item['tags'] = array_values(array_filter(array_map('trim', explode(',', $_POST['tags'] ?? implode(',', $item['tags'] ?? [])))));
            $item['location'] = trim($_POST['location'] ?? $item['location']);
            $item['offerType'] = $_POST['offerType'] ?? ($item['offerType'] ?? 'venta');
            $item['shipping'] = isset($_POST['shipping']) ? 1 : 0;
            if (($item['status'] ?? '') === 'approved' && !$this->auth->hasRole(['admin', 'superadmin'])) {
                $item['status'] = 'pending';
            }
        }

        if (!empty($_FILES['images']['name'][0])) {
            $new = $this->handleImage($_FILES['images'], 'listings/' . $item['id'] . '/images', $this->config['app']['max_images_per_listing']);
            if ($new) {
                $item['images'] = array_slice(array_merge($item['images'] ?? [], $new), 0, $this->config['app']['max_images_per_listing']);
            }
        }

        $this->market->saveListing($item);
        Helpers::flash('success', 'Publicación actualizada.');
        Helpers::redirect('/panel/publicaciones');
    }

    public function listingDelete(): void
    {
        $this->auth->requireRole(['user', 'admin', 'superadmin']);
        if (!$this->security->verifyCsrf($_POST['_csrf'] ?? null)) Helpers::redirect('/panel/publicaciones');
        $id = $_GET['id'] ?? '';
        $item = $this->market->getListing($id);
        $user = $this->auth->user();
        if (!$item) Helpers::redirect('/panel/publicaciones');
        $can = $this->auth->hasRole(['admin', 'superadmin']) || ($item['ownerId'] === $user['id'] && in_array($item['status'], ['pending', 'rejected'], true));
        if (!$can) {
            Helpers::flash('error', 'No tienes permisos para eliminar esta publicación.');
            Helpers::redirect('/panel/publicaciones');
        }
        $this->market->deleteListing($id);
        Helpers::flash('success', 'Publicación eliminada.');
        Helpers::redirect('/panel/publicaciones');
    }

    private function handleImage(?array $file, string $relativeDir, int $limit): array
    {
        if (!$file) return [];
        $out = [];
        $files = [];
        if (is_array($file['name'])) {
            foreach ($file['name'] as $i => $n) {
                if ($file['error'][$i] === UPLOAD_ERR_OK) {
                    $files[] = [
                        'name' => $n,
                        'tmp_name' => $file['tmp_name'][$i],
                        'size' => $file['size'][$i],
                    ];
                }
            }
        } else {
            if (($file['error'] ?? 1) === UPLOAD_ERR_OK) {
                $files[] = ['name' => $file['name'], 'tmp_name' => $file['tmp_name'], 'size' => $file['size'] ?? 0];
            }
        }

        $destDir = $this->storage->path($relativeDir);
        if (!is_dir($destDir)) mkdir($destDir, 0775, true);
        $finfo = finfo_open(FILEINFO_MIME_TYPE);

        foreach ($files as $f) {
            if (count($out) >= $limit) break;
            if ($f['size'] > 6 * 1024 * 1024) continue;
            $mime = finfo_file($finfo, $f['tmp_name']) ?: '';
            if (!in_array($mime, ['image/jpeg', 'image/png', 'image/webp'], true)) continue;
            if (!@getimagesize($f['tmp_name'])) continue;
            $ext = $mime === 'image/png' ? 'png' : ($mime === 'image/webp' ? 'webp' : 'jpg');
            $name = bin2hex(random_bytes(8)) . '.' . $ext;
            $target = $destDir . DIRECTORY_SEPARATOR . $name;
            if (!move_uploaded_file($f['tmp_name'], $target)) {
                continue;
            }
            $out[] = $name;
        }
        if (is_resource($finfo)) finfo_close($finfo);
        return $out;
    }
}
