<?php
namespace App\Controllers;

use App\Core\Controller;
use App\Core\Helpers;

class SuperAdminController extends Controller
{
    public function admins(): void
    {
        $this->auth->requireRole(['superadmin']);
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            if (!$this->security->verifyCsrf($_POST['_csrf'] ?? null)) Helpers::redirect('/superadmin/admins');
            $this->auth->register([
                'name' => trim($_POST['name'] ?? 'Admin'),
                'email' => strtolower(trim($_POST['email'] ?? '')),
                'phone' => '',
                'whatsapp' => '',
                'location' => '',
                'password' => $_POST['password'] ?? 'Cambiar123!',
                'role' => 'admin',
                'forcePasswordChange' => true,
            ]);
            Helpers::redirect('/superadmin/admins');
        }

        $users = $this->storage->readJson('users/users.json', ['items' => []])['items'] ?? [];
        $admins = array_values(array_filter($users, fn($u) => in_array($u['role'], ['admin', 'superadmin'], true)));
        $this->view('superadmin/admins', ['title' => 'Gestión de admins', 'admins' => $admins]);
    }

    public function backupCreate(): void
    {
        $this->auth->requireRole(['superadmin']);
        if (!$this->security->verifyCsrf($_POST['_csrf'] ?? null)) Helpers::redirect('/superadmin/admins');

        $source = realpath($this->storage->path('.'));
        $backupPath = $this->storage->path('backups/backup_' . date('Ymd_His') . '.zip');

        $zip = new \ZipArchive();
        if ($zip->open($backupPath, \ZipArchive::CREATE) === true) {
            $it = new \RecursiveIteratorIterator(new \RecursiveDirectoryIterator($source, \RecursiveDirectoryIterator::SKIP_DOTS));
            foreach ($it as $file) {
                $filePath = $file->getRealPath();
                if (str_contains($filePath, DIRECTORY_SEPARATOR . 'backups' . DIRECTORY_SEPARATOR)) continue;
                $local = substr($filePath, strlen($source) + 1);
                $zip->addFile($filePath, $local);
            }
            $zip->close();
            $this->storage->appendLog('admin.log', 'BACKUP_CREATED ' . basename($backupPath));
            Helpers::flash('success', 'Backup creado: ' . basename($backupPath));
        } else {
            Helpers::flash('error', 'No se pudo crear el backup.');
        }
        Helpers::redirect('/superadmin/admins');
    }

    public function backupRestore(): void
    {
        $this->auth->requireRole(['superadmin']);
        if (!$this->security->verifyCsrf($_POST['_csrf'] ?? null)) Helpers::redirect('/superadmin/admins');
        $file = trim($_POST['file'] ?? '');
        if (!$file) {
            Helpers::flash('error', 'Selecciona un backup.');
            Helpers::redirect('/superadmin/admins');
        }
        $backup = $this->storage->path('backups/' . basename($file));
        if (!file_exists($backup)) {
            Helpers::flash('error', 'Backup no encontrado.');
            Helpers::redirect('/superadmin/admins');
        }

        $zip = new \ZipArchive();
        if ($zip->open($backup) === true) {
            $zip->extractTo($this->storage->path('.'));
            $zip->close();
            Helpers::flash('success', 'Backup restaurado. Revisa integridad.');
        } else {
            Helpers::flash('error', 'No se pudo restaurar.');
        }
        Helpers::redirect('/superadmin/admins');
    }

    public function config(): void
    {
        $this->auth->requireRole(['superadmin']);
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            if (!$this->security->verifyCsrf($_POST['_csrf'] ?? null)) Helpers::redirect('/superadmin/config');
            $settings = $this->storage->readJson('pages/settings.json', []);
            $settings['maintenance_mode'] = isset($_POST['maintenance_mode']);
            $settings['site_name'] = trim($_POST['site_name'] ?? 'Venta y Compra - Taller Mecánico Mojarra');
            $settings['mediator_phone'] = trim($_POST['mediator_phone'] ?? '');
            $settings['mediator_whatsapp'] = trim($_POST['mediator_whatsapp'] ?? '');
            $settings['default_contact_mode'] = in_array($_POST['default_contact_mode'] ?? '', ['seller', 'mediator'])
                ? $_POST['default_contact_mode'] : 'seller';
            $this->storage->writeJson('pages/settings.json', $settings);
            Helpers::flash('success', 'Configuración guardada.');
            Helpers::redirect('/superadmin/config');
        }

        $settings = $this->storage->readJson('pages/settings.json', ['maintenance_mode' => false, 'site_name' => 'Venta y Compra - Taller Mecánico Mojarra']);
        $backups = glob($this->storage->path('backups/*.zip')) ?: [];
        $this->view('superadmin/config', ['title' => 'Configuración global', 'settings' => $settings, 'backups' => array_map('basename', $backups)]);
    }

    public function resetPassword(): void
    {
        $this->auth->requireRole(['superadmin']);
        if (!$this->security->verifyCsrf($_POST['_csrf'] ?? null)) Helpers::redirect('/superadmin/admins');
        $userId = trim($_POST['user_id'] ?? '');
        $newPass = trim($_POST['new_password'] ?? '');
        if (!$userId || strlen($newPass) < 8) {
            Helpers::flash('error', 'Datos inválidos para reset.');
            Helpers::redirect('/superadmin/admins');
        }
        $sec = $this->storage->readJson('users/' . $userId . '/security.json', []);
        if (!$sec) {
            Helpers::flash('error', 'Usuario no encontrado.');
            Helpers::redirect('/superadmin/admins');
        }
        $sec['passwordHash'] = password_hash($newPass, PASSWORD_BCRYPT);
        $sec['lastPasswordChange'] = time();
        $this->storage->writeJson('users/' . $userId . '/security.json', $sec);
        Helpers::flash('success', 'Contraseña reseteada para ' . $userId);
        Helpers::redirect('/superadmin/admins');
    }
}
