<?php
namespace App\Core;

class Auth
{
    public function __construct(private Storage $storage, private array $config)
    {
    }

    public function user(): ?array
    {
        $id = $_SESSION['user_id'] ?? null;
        if (!$id) {
            return null;
        }
        return $this->storage->readJson('users/' . $id . '/profile.json', []);
    }

    public function check(): bool
    {
        return !empty($_SESSION['user_id']);
    }

    public function attempt(string $email, string $password): bool
    {
        $usersIndex = $this->storage->readJson('users/users.json', ['items' => []]);
        foreach ($usersIndex['items'] as $item) {
            if (strcasecmp($item['email'], $email) === 0) {
                if (($item['status'] ?? 'active') !== 'active') {
                    return false;
                }
                $security = $this->storage->readJson('users/' . $item['id'] . '/security.json', []);
                if (!empty($security['passwordHash']) && password_verify($password, $security['passwordHash'])) {
                    $_SESSION['user_id'] = $item['id'];
                    $this->storage->appendLog('auth.log', 'LOGIN_OK ' . $item['id'] . ' IP:' . ($_SERVER['REMOTE_ADDR'] ?? 'n/a'));
                    return true;
                }
            }
        }
        $this->storage->appendLog('auth.log', 'LOGIN_FAIL ' . $email . ' IP:' . ($_SERVER['REMOTE_ADDR'] ?? 'n/a'));
        return false;
    }

    public function logout(): void
    {
        unset($_SESSION['user_id']);
        session_regenerate_id(true);
    }

    public function hasRole(array $roles): bool
    {
        $user = $this->user();
        return $user && in_array($user['role'] ?? 'user', $roles, true);
    }

    public function requireRole(array $roles): void
    {
        if (!$this->check()) {
            Helpers::flash('error', 'Debes iniciar sesión.');
            Helpers::redirect('/login');
        }
        if (!$this->hasRole($roles)) {
            http_response_code(403);
            exit('403 - Sin permisos');
        }
    }

    public function register(array $data): string
    {
        $id = Helpers::randomId('u');
        $profile = [
            'id' => $id,
            'role' => $data['role'] ?? 'user',
            'name' => $data['name'],
            'email' => strtolower($data['email']),
            'phone' => $data['phone'] ?? '',
            'whatsapp' => $data['whatsapp'] ?? '',
            'location' => $data['location'] ?? '',
            'avatar' => '',
            'createdAt' => time(),
            'status' => 'active',
            'verified' => false,
            'forcePasswordChange' => $data['forcePasswordChange'] ?? false,
        ];
        $security = [
            'passwordHash' => password_hash($data['password'], $this->config['security']['password_algo']),
            'resetTokens' => [],
            'lastPasswordChange' => time(),
        ];

        $this->storage->writeJson('users/' . $id . '/profile.json', $profile);
        $this->storage->writeJson('users/' . $id . '/security.json', $security);

        $index = $this->storage->readJson('users/users.json', ['items' => []]);
        $index['items'][] = [
            'id' => $id,
            'email' => $profile['email'],
            'name' => $profile['name'],
            'role' => $profile['role'],
            'status' => $profile['status'],
            'createdAt' => $profile['createdAt'],
        ];
        $this->storage->writeJson('users/users.json', $index);

        return $id;
    }
}
