<?php
namespace App\Core;

class Marketplace
{
    public function __construct(private Storage $storage)
    {
    }

    public function getCategories(): array
    {
        $data = $this->storage->readJson('categories/categories.json', ['items' => []]);
        return $data['items'] ?? [];
    }

    public function saveCategories(array $items): void
    {
        $this->storage->writeJson('categories/categories.json', ['items' => array_values($items)]);
    }

    public function allListings(): array
    {
        $index = $this->storage->readJson('listings/listings.json', ['items' => []]);
        return $index['items'] ?? [];
    }

    public function getListing(string $id): ?array
    {
        $data = $this->storage->readJson('listings/' . $id . '/data.json', []);
        return $data ?: null;
    }

    public function saveListing(array $listing): string
    {
        $id = $listing['id'] ?? Helpers::randomId('l');
        $listing['id'] = $id;
        $listing['updatedAt'] = time();
        if (!isset($listing['createdAt'])) {
            $listing['createdAt'] = time();
        }
        $this->storage->writeJson('listings/' . $id . '/data.json', $listing);

        $idx = $this->storage->readJson('listings/listings.json', ['items' => []]);
        $found = false;
        foreach ($idx['items'] as &$row) {
            if ($row['id'] === $id) {
                $row = $this->toListingIndex($listing);
                $found = true;
                break;
            }
        }
        if (!$found) {
            $idx['items'][] = $this->toListingIndex($listing);
        }
        $this->storage->writeJson('listings/listings.json', $idx);
        return $id;
    }

    public function deleteListing(string $id): void
    {
        $idx = $this->storage->readJson('listings/listings.json', ['items' => []]);
        $idx['items'] = array_values(array_filter($idx['items'], fn($x) => $x['id'] !== $id));
        $this->storage->writeJson('listings/listings.json', $idx);
        $dir = $this->storage->path('listings/' . $id);
        if (is_dir($dir)) {
            $this->deleteDir($dir);
        }
    }

    public function searchListings(array $filters): array
    {
        $items = $this->allListings();
        $q = strtolower(trim((string)($filters['q'] ?? '')));
        $cat = strtolower((string)($filters['cat'] ?? ''));
        $cond = strtolower((string)($filters['cond'] ?? ''));
        $loc = strtolower((string)($filters['loc'] ?? ''));
        $status = strtolower((string)($filters['status'] ?? 'approved'));
        $min = is_numeric($filters['min'] ?? null) ? (float)$filters['min'] : null;
        $max = is_numeric($filters['max'] ?? null) ? (float)$filters['max'] : null;
        $type = strtolower((string)($filters['type'] ?? ''));
        $shipping = (string)($filters['shipping'] ?? '');
        $verified = (string)($filters['verified'] ?? '');

        $out = array_filter($items, function ($it) use ($q, $cat, $cond, $loc, $status, $min, $max, $type, $shipping, $verified) {
            if ($status === 'visible') {
                if (!in_array($it['status'] ?? '', ['approved', 'sold'])) return false;
            } elseif ($status && ($it['status'] ?? '') !== $status && $status !== 'all') return false;
            if ($q && !str_contains(strtolower(($it['title'] ?? '') . ' ' . implode(' ', $it['tags'] ?? [])), $q)) return false;
            if ($cat && strtolower($it['category'] ?? '') !== $cat) return false;
            if ($cond && strtolower($it['condition'] ?? '') !== $cond) return false;
            if ($loc && !str_contains(strtolower($it['location'] ?? ''), $loc)) return false;
            if ($min !== null && (float)($it['price'] ?? 0) < $min) return false;
            if ($max !== null && (float)($it['price'] ?? 0) > $max) return false;
            if ($type && strtolower($it['offerType'] ?? 'venta') !== $type) return false;
            if ($shipping !== '' && (int)($it['shipping'] ?? 0) !== (int)$shipping) return false;
            if ($verified !== '' && (int)($it['sellerVerified'] ?? 0) !== (int)$verified) return false;
            return true;
        });

        usort($out, function ($a, $b) {
            if (($a['featured'] ?? false) !== ($b['featured'] ?? false)) {
                return ($a['featured'] ?? false) ? -1 : 1;
            }
            return ($b['updatedAt'] ?? 0) <=> ($a['updatedAt'] ?? 0);
        });

        return array_values($out);
    }

    public function incrementViews(string $id): void
    {
        $item = $this->getListing($id);
        if (!$item) return;
        $item['views'] = (int)($item['views'] ?? 0) + 1;
        $this->saveListing($item);
    }

    public function saveReport(array $report): void
    {
        $data = $this->storage->readJson('reports/reports.json', ['items' => []]);
        $report['id'] = Helpers::randomId('r');
        $report['createdAt'] = time();
        $report['status'] = 'open';
        $data['items'][] = $report;
        $this->storage->writeJson('reports/reports.json', $data);
    }

    public function reports(): array
    {
        return $this->storage->readJson('reports/reports.json', ['items' => []])['items'] ?? [];
    }

    public function saveAds(array $slots): void
    {
        $this->storage->writeJson('ads/ad_slots.json', ['slots' => array_values($slots)]);
    }

    public function ads(): array
    {
        return $this->storage->readJson('ads/ad_slots.json', ['slots' => []])['slots'] ?? [];
    }

    public function pages(): array
    {
        return $this->storage->readJson('pages/pages.json', ['items' => []])['items'] ?? [];
    }

    public function savePages(array $items): void
    {
        $this->storage->writeJson('pages/pages.json', ['items' => array_values($items)]);
    }

    public function stats(): array
    {
        $users = $this->storage->readJson('users/users.json', ['items' => []])['items'] ?? [];
        $listings = $this->allListings();
        $reports = $this->reports();
        $today = strtotime('today');

        return [
            'pending' => count(array_filter($listings, fn($l) => ($l['status'] ?? '') === 'pending')),
            'approvedToday' => count(array_filter($listings, fn($l) => ($l['status'] ?? '') === 'approved' && ($l['updatedAt'] ?? 0) >= $today)),
            'newUsers' => count(array_filter($users, fn($u) => ($u['createdAt'] ?? 0) >= $today)),
            'reports' => count(array_filter($reports, fn($r) => ($r['status'] ?? '') === 'open')),
            'totalListings' => count($listings),
            'sold' => count(array_filter($listings, fn($l) => ($l['status'] ?? '') === 'sold')),
        ];
    }

    public function rebuildListingIndex(): void
    {
        $base = $this->storage->path('listings');
        $items = [];
        if (!is_dir($base)) {
            return;
        }
        foreach (scandir($base) as $dir) {
            if (!str_starts_with($dir, 'l_')) continue;
            $data = $this->storage->readJson('listings/' . $dir . '/data.json', []);
            if ($data) {
                $items[] = $this->toListingIndex($data);
            }
        }
        $this->storage->writeJson('listings/listings.json', ['items' => $items]);
    }

    private function toListingIndex(array $listing): array
    {
        return [
            'id' => $listing['id'],
            'ownerId' => $listing['ownerId'],
            'title' => $listing['title'],
            'slug' => $listing['slug'],
            'category' => $listing['category'],
            'condition' => $listing['condition'],
            'price' => $listing['price'],
            'location' => $listing['location'],
            'status' => $listing['status'],
            'featured' => (bool)($listing['featured'] ?? false),
            'sellerVerified' => (bool)($listing['sellerVerified'] ?? false),
            'shipping' => (int)($listing['shipping'] ?? 0),
            'offerType' => $listing['offerType'] ?? 'venta',
            'tags' => $listing['tags'] ?? [],
            'image' => $listing['images'][0] ?? '',
            'createdAt' => $listing['createdAt'],
            'updatedAt' => $listing['updatedAt'],
            'views' => $listing['views'] ?? 0,
            'contactMode' => $listing['contactMode'] ?? 'seller',
        ];
    }

    private function deleteDir(string $path): void
    {
        $it = new \RecursiveDirectoryIterator($path, \RecursiveDirectoryIterator::SKIP_DOTS);
        $files = new \RecursiveIteratorIterator($it, \RecursiveIteratorIterator::CHILD_FIRST);
        foreach ($files as $file) {
            $file->isDir() ? rmdir($file->getRealPath()) : unlink($file->getRealPath());
        }
        rmdir($path);
    }
}
