<?php
namespace App\Core;

class Storage
{
    public function __construct(private array $config)
    {
    }

    public function path(string $relative): string
    {
        return rtrim($this->config['paths']['storage'], '/\\') . DIRECTORY_SEPARATOR . ltrim($relative, '/\\');
    }

    public function ensureDir(string $dir): void
    {
        if (!is_dir($dir)) {
            mkdir($dir, 0775, true);
        }
    }

    public function readJson(string $relative, array $default = []): array
    {
        $path = $this->path($relative);
        if (!file_exists($path)) {
            return $default;
        }
        $json = file_get_contents($path);
        if ($json === false || $json === '') {
            return $default;
        }
        $decoded = json_decode($json, true);
        return is_array($decoded) ? $decoded : $default;
    }

    public function writeJson(string $relative, array $data): bool
    {
        $path = $this->path($relative);
        $this->ensureDir(dirname($path));
        $fp = fopen($path, 'c+');
        if (!$fp) {
            return false;
        }
        if (!flock($fp, LOCK_EX)) {
            fclose($fp);
            return false;
        }

        $encoded = json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        $ok = $encoded !== false;
        if ($ok) {
            rewind($fp);
            $ok = ftruncate($fp, 0);
        }
        if ($ok) {
            $ok = fwrite($fp, $encoded . PHP_EOL) !== false;
        }
        if ($ok) {
            fflush($fp);
        }
        flock($fp, LOCK_UN);
        fclose($fp);
        return $ok;
    }

    public function appendLog(string $file, string $message): void
    {
        $path = $this->path('logs/' . $file);
        $this->ensureDir(dirname($path));
        $line = '[' . date('Y-m-d H:i:s') . '] ' . $message . PHP_EOL;
        file_put_contents($path, $line, FILE_APPEND | LOCK_EX);
    }

    public function withLock(string $relative, callable $callback)
    {
        $path = $this->path($relative);
        $this->ensureDir(dirname($path));
        $fp = fopen($path, 'c+');
        if (!$fp) {
            return null;
        }
        try {
            flock($fp, LOCK_EX);
            return $callback($fp, $path);
        } finally {
            flock($fp, LOCK_UN);
            fclose($fp);
        }
    }
}
