/**
 * app.js - Main JavaScript for Mojarra Ventas Marketplace
 * Vanilla JS, no dependencies.
 */
document.addEventListener('DOMContentLoaded', () => {

  /* ==========================================================
     1. MOBILE MENU TOGGLE
     ========================================================== */
  const menuToggle = document.getElementById('menuToggle');
  const navLinks   = document.getElementById('navLinks');

  if (menuToggle && navLinks) {
    menuToggle.addEventListener('click', (e) => {
      e.stopPropagation();
      navLinks.classList.toggle('active');
    });

    document.addEventListener('click', (e) => {
      if (!navLinks.contains(e.target) && !menuToggle.contains(e.target)) {
        navLinks.classList.remove('active');
      }
    });
  }

  /* ==========================================================
     2. CATEGORIES MEGA MENU
     ========================================================== */
  const catDropdown = document.querySelector('.nav-categories-dropdown');

  if (catDropdown) {
    const catButton = catDropdown.querySelector('button');
    const megaMenu  = catDropdown.querySelector('.mega-menu');

    if (catButton && megaMenu) {
      catButton.addEventListener('click', (e) => {
        e.stopPropagation();
        megaMenu.classList.toggle('active');
      });

      document.addEventListener('click', (e) => {
        if (!catDropdown.contains(e.target)) {
          megaMenu.classList.remove('active');
        }
      });
    }
  }

  /* ==========================================================
     3. CONFIRM DIALOGS
     ========================================================== */
  document.querySelectorAll('form[data-confirm]').forEach((form) => {
    form.addEventListener('submit', (e) => {
      const message = form.getAttribute('data-confirm');
      if (!confirm(message)) {
        e.preventDefault();
      }
    });
  });

  /* ==========================================================
     4. FLASH MESSAGE AUTO-DISMISS
     ========================================================== */
  const flashMsg = document.getElementById('flashMsg');

  if (flashMsg) {
    setTimeout(() => {
      flashMsg.style.transition = 'opacity 0.5s ease';
      flashMsg.style.opacity = '0';
      flashMsg.addEventListener('transitionend', () => {
        flashMsg.remove();
      }, { once: true });
    }, 5000);
  }

  /* ==========================================================
     5. MODAL SYSTEM
     ========================================================== */
  const modalOverlay = document.getElementById('modalOverlay');
  const modalClose   = document.getElementById('modalClose');
  const modalBody    = document.getElementById('modalBody');

  /**
   * Open a modal with the given HTML content.
   * Exposed globally so other scripts / inline handlers can call it.
   * @param {string} htmlContent
   */
  window.openModal = function openModal(htmlContent) {
    if (!modalOverlay) return;

    if (modalBody) {
      modalBody.innerHTML = htmlContent;
    }
    modalOverlay.classList.add('active');
  };

  /**
   * Close the currently open modal.
   */
  window.closeModal = function closeModal() {
    if (!modalOverlay) return;
    modalOverlay.classList.remove('active');
  };

  if (modalClose) {
    modalClose.addEventListener('click', window.closeModal);
  }

  if (modalOverlay) {
    modalOverlay.addEventListener('click', (e) => {
      if (e.target === modalOverlay) {
        window.closeModal();
      }
    });
  }

  /* ==========================================================
     6. NAVBAR SCROLL EFFECT
     ========================================================== */
  const navbar = document.getElementById('navbar');

  if (navbar) {
    const applyScrollClass = () => {
      if (window.scrollY > 10) {
        navbar.classList.add('scrolled');
      } else {
        navbar.classList.remove('scrolled');
      }
    };

    // Apply on load in case page is already scrolled (e.g. after refresh).
    applyScrollClass();
    window.addEventListener('scroll', applyScrollClass, { passive: true });
  }

  /* ==========================================================
     7. IMAGE PREVIEW FOR FILE INPUTS
     ========================================================== */
  const imageInputs = document.querySelectorAll(
    'input[type="file"][name="images[]"], input[type="file"][name="avatar"]'
  );

  imageInputs.forEach((input) => {
    input.addEventListener('change', () => {
      // Find or create the preview container right after the input.
      let preview = input.parentElement.querySelector('.image-preview-container');
      if (!preview) {
        preview = document.createElement('div');
        preview.className = 'image-preview-container';
        preview.style.display = 'flex';
        preview.style.flexWrap = 'wrap';
        preview.style.gap = '8px';
        preview.style.marginTop = '8px';
        input.parentElement.appendChild(preview);
      }

      // Clear previous previews.
      preview.innerHTML = '';

      const files = input.files;
      if (!files || files.length === 0) return;

      Array.from(files).forEach((file) => {
        if (!file.type.startsWith('image/')) return;

        const reader = new FileReader();
        reader.addEventListener('load', () => {
          const img = document.createElement('img');
          img.src = reader.result;
          img.alt = 'Preview';
          img.style.width = '100px';
          img.style.height = '100px';
          img.style.objectFit = 'cover';
          img.style.borderRadius = '6px';
          img.style.border = '1px solid #ddd';
          preview.appendChild(img);
        });
        reader.readAsDataURL(file);
      });
    });
  });

  /* ==========================================================
     8. SHARE BUTTON
     ========================================================== */
  document.querySelectorAll('[data-share]').forEach((btn) => {
    btn.addEventListener('click', async () => {
      const shareUrl  = btn.getAttribute('data-share') || window.location.href;
      const shareData = {
        title: document.title,
        url: shareUrl,
      };

      if (navigator.share) {
        try {
          await navigator.share(shareData);
        } catch (err) {
          // User cancelled or API error - silently ignore.
        }
      } else {
        try {
          await navigator.clipboard.writeText(shareUrl);
          alert('Enlace copiado al portapapeles.');
        } catch (err) {
          // Clipboard API not available either; prompt-based fallback.
          prompt('Copia este enlace:', shareUrl);
        }
      }
    });
  });

  /* ==========================================================
     9. GALLERY FUNCTIONALITY (Product Detail)
     ========================================================== */
  const galleryThumbs = document.querySelectorAll('.gallery-thumb');
  const galleryMain   = document.querySelector('.gallery-main img');

  if (galleryThumbs.length && galleryMain) {
    galleryThumbs.forEach((thumb) => {
      thumb.addEventListener('click', () => {
        // Update main image source.
        const newSrc = thumb.getAttribute('src') ||
                       thumb.getAttribute('data-src') ||
                       thumb.querySelector('img')?.getAttribute('src');

        if (newSrc) {
          galleryMain.src = newSrc;
        }

        // Toggle active class: remove from siblings, add to clicked.
        galleryThumbs.forEach((t) => t.classList.remove('active'));
        thumb.classList.add('active');
      });
    });
  }

  /* ==========================================================
     10. LAZY LOADING (IntersectionObserver)
     ========================================================== */
  const lazyImages = document.querySelectorAll('img[data-src]');

  if (lazyImages.length && 'IntersectionObserver' in window) {
    const lazyObserver = new IntersectionObserver((entries, observer) => {
      entries.forEach((entry) => {
        if (entry.isIntersecting) {
          const img = entry.target;
          img.src = img.getAttribute('data-src');

          // Also handle data-srcset if present.
          const srcset = img.getAttribute('data-srcset');
          if (srcset) {
            img.srcset = srcset;
          }

          img.removeAttribute('data-src');
          img.removeAttribute('data-srcset');
          observer.unobserve(img);
        }
      });
    }, {
      rootMargin: '200px 0px', // Start loading a bit before they enter the viewport.
    });

    lazyImages.forEach((img) => lazyObserver.observe(img));
  } else {
    // Fallback: load all images immediately if IntersectionObserver is unavailable.
    lazyImages.forEach((img) => {
      img.src = img.getAttribute('data-src');
      img.removeAttribute('data-src');
    });
  }

  /* ==========================================================
     11. SMOOTH SCROLL FOR ANCHOR LINKS
     ========================================================== */
  document.querySelectorAll('a[href^="#"]').forEach((anchor) => {
    anchor.addEventListener('click', (e) => {
      const targetId = anchor.getAttribute('href');
      if (targetId === '#' || targetId === '') return;

      const target = document.querySelector(targetId);
      if (target) {
        e.preventDefault();
        target.scrollIntoView({ behavior: 'smooth', block: 'start' });

        // Close mobile menu if open after navigating.
        if (navLinks) {
          navLinks.classList.remove('active');
        }
      }
    });
  });

  /* ==========================================================
     12. TAB SYSTEM (Admin Pages)
     ========================================================== */
  const tabButtons = document.querySelectorAll('.tab-btn');
  const tabPanels  = document.querySelectorAll('.tab-panel');

  if (tabButtons.length && tabPanels.length) {
    tabButtons.forEach((btn) => {
      btn.addEventListener('click', () => {
        const targetTab = btn.getAttribute('data-tab');
        if (!targetTab) return;

        // Deactivate all buttons and panels.
        tabButtons.forEach((b) => b.classList.remove('active'));
        tabPanels.forEach((p) => p.classList.remove('active'));

        // Activate clicked button and matching panel.
        btn.classList.add('active');

        const targetPanel = document.querySelector(
          `.tab-panel[data-tab="${targetTab}"]`
        );
        if (targetPanel) {
          targetPanel.classList.add('active');
        }
      });
    });
  }

});
